package net.sf.saxon.tinytree.extension;

import net.sf.saxon.Configuration;
import net.sf.saxon.om.NodeInfo;
import net.sf.saxon.tinytree.TinyTree;

/**
 * 
 * interface to implement by a {@link Configuration} instance to support the following behaviors : 
 * - allow to perform some evaluation on a TinyTree's attributes/text nodes (see {@link IValueResolver})
 * - allow to support different modes on a TinyTree's attributes values (see {@link IAttributePlaceHolder})  
 * 
 * 
 * @author lcamus
 *
 */
public interface IExtensionSupport {

    /**
     * only for internal usage
     * @author lcamus
     *
     */
    public interface IPlaceHolderListener {
	void handlePlaceHolderSupportEnabled(boolean enable);
    }
    
    /**
     * simple interface to easily access a TinyTree from its contained objects.
     * should not be implemented, only accessed
     * @author lcamus
     *
     */
    public interface ITinyTreeAdapter {

	TinyTree getTinyTree();
    }

    /**
     * To implement by a class able to perform some interpretation of a specific attribute/text value in a tinyTree
     * @author lcamus
     *
     */
    public interface IValueResolver {
	/**
	 * 
	 * @param node
	 * @param currentAttributeNodeNr
	 * @return the value of the specified attribute
	 */
	public CharSequence readAttribute(NodeInfo node, int currentAttributeNodeNr);
	/**
	 * 
	 * @param tree
	 * @param nodeNr
	 * @return the value of the specified textNode
	 */
	public CharSequence readTextNode(TinyTree tree, int nodeNr);
	
	/**
	 * 
	 * @param value : the original value of from the read XML tinyTree
	 * @return true if the value can be resolved by this instance. If not, the XML values will be the one read
	 */
	public boolean resolveValue(CharSequence value);
    }
    
    /**
     * implementation should allow to change the CharSequence of the registered CharSequence by another when enabled.
     * 
     * The implementation is responsible to adapt/restore the registered CharSequence of the attributes 
     * which returned true for {@link #handleValue(CharSequence)}
     * @author lcamus
     *
     */
    public interface IAttributePlaceHolder {
	/**
	 * register the specified attribute from the underlyning TinyTree
	 * @param parent
	 * @param attributeNumber
	 * @param value : the original value of the attribute
	 */
	public void registerAttribute(int parent, int attributeNumber, CharSequence value);
	
	/**
	 * apply/restore the replacement/orginal value on the registered attributes.
	 * Must not be called 
	 * @param enabled
	 */
	public void apply(boolean enabled);
	
	/**
	 * 
	 * @param value
	 * @return true if this originalValue can be handled by this instance
	 */
	public boolean handleValue(CharSequence value);
	
	/**
	 * method to store the underlying TinyTree which we will change. 
	 * This method must not be called
	 * @param tinyTree
	 */
	public void setTinyTree(TinyTree tinyTree);
    }

    public void addListener(IPlaceHolderListener listener);

    public void removeListener(IPlaceHolderListener listener);
    
    /**
     * enable/disable the {@link IAttributePlaceHolder}.
     * the implementation should call {@link IAttributePlaceHolder#apply(boolean)} to perform the update on the TinyTree
     * @param enabled
     */
    public void enablePlaceHolderSupport(boolean enabled);

    /**
     * 
     * @return the valueResolver which will be used to intrepret specific syntaxes on an attribute/textNode value in the tiny tree
     * as ValueResolver are shared at the configuration level, they must not change the TinyTree content in anyway and retains
     * index informations because he same instance may be against different XML TinyTree 
     */
    IValueResolver getValueResolver();

    
    IAttributePlaceHolder createAttributePlaceHolder();

}
